#include "MyProject2_objects.h"
#include "MyProject2_resources.h"
#include "built_in.h"
#include "FT800_Types.h"
#include "ILI_cmds.h"

/*TFT module connections
sbit FT800_RST at LATB.B2;
sbit FT800_CS at LATB.B1;
sbit FT800_RST_Direction at TRISB.B2;
sbit FT800_CS_Direction at TRISB.B1;
*/

// Object type constants
// Usage: VTFT stack internally
const VTFT_OT_IMAGE      = 6;
const VTFT_OT_CIRCLE     = 8;
const VTFT_OT_BOX        = 12;
const VTFT_OT_EVESPINNER = 33;
// ~Object type constants

// Event type constants
// Usage: OnEvent
const VTFT_EVT_UP    = 0;
const VTFT_EVT_DOWN  = 1;
const VTFT_EVT_CLICK = 2;
const VTFT_EVT_PRESS = 3;
// ~Event type constants

// Sound action constants
// Usage: sound event action property and ProcessEvent
const VTFT_SNDACT_NONE  = 0;
const VTFT_SNDACT_PLAY  = 1;
const VTFT_SNDACT_STOP  = 2;
// ~Sound action constants

// Resource loading constants.
// Usage: DrawScreenO and LoadCurrentScreenResToGRAM
const VTFT_LOAD_RES_NONE    = 0x00; // do not load g-ram resources
const VTFT_LOAD_RES_STATIC  = 0x01; // load g-ram resources for static objects
const VTFT_LOAD_RES_DYNAMIC = 0x02; // load g-ram resources for dynamic objects
const VTFT_LOAD_RES_ALL     = 0x03; // load g-ram resources for all objects
// ~Resource loading constants

// Display effect constants
// Usage: DrawScreenO
const VTFT_DISPLAY_EFF_NONE         = 0x00; // no effect when switching between screens
const VTFT_DISPLAY_EFF_LIGHTS_FADE  = 0x04; // backlight: fade out before, fade in after drawing new screen
const VTFT_DISPLAY_EFF_LIGHTS_OFF   = 0x08; // backlight: turn off before, turn on after drawing new screen
// ~Display effect constants

// Stack flags
// Usage: internally used by VTFT stack
const VTFT_INT_REPAINT_ON_DOWN     = 1 << 0;
const VTFT_INT_REPAINT_ON_UP       = 1 << 1;
const VTFT_INT_BRING_TO_FRONT      = 1 << 2;
const VTFT_INT_INTRINSIC_CLICK_EFF = 1 << 3;
// ~Stack flags

// Table of object draw handlers
// Use object type constants to access coresponding object draw handler
const TPointer DrawHandlerTable[44] = {
  0,               // Button draw handler not used
  0,               // CButton draw handler not used
  0,               // ButtonRound draw handler not used
  0,               // CButtonRound draw handler not used
  0,               // Label draw handler not used
  0,               // CLabel draw handler not used
  &DrawImage,      // Image draw handler
  0,               // CImage draw handler not used
  &DrawCircle,     // Circle draw handler
  0,               // CCircle draw handler not used
  0,               // CircleButton draw handler not used
  0,               // CCircleButton draw handler not used
  &DrawBox,        // Box draw handler
  0,               // CBox draw handler not used
  0,               // BoxRound draw handler not used
  0,               // CBoxRound draw handler not used
  0,               // Line draw handler not used
  0,               // CLine draw handler not used
  0,               // Polygon draw handler not used
  0,               // CPolygon draw handler not used
  0,               // CheckBox draw handler not used
  0,               // RadioButton draw handler not used
  0,               // ProgressBar draw handler not used
  0,               // Audio draw handler not used
  0,               // EveClock draw handler not used
  0,               // EveGauge draw handler not used
  0,               // EveDial draw handler not used
  0,               // EveKeys draw handler not used
  0,               // CEveKeys draw handler not used
  0,               // EveProgressBar draw handler not used
  0,               // EveScrollBar draw handler not used
  0,               // EveToggle draw handler not used
  0,               // EveSlider draw handler not used
  &DrawEveSpinner, // EveSpinner draw handler
  0,               // EveScreenSaver draw handler not used
  0,               // EveSketch draw handler not used
  0,               // EveButton draw handler not used
  0,               // CEveButton draw handler not used
  0,               // EveGradient draw handler not used
  0,               // CEveGradient draw handler not used
  0,               // EveText draw handler not used
  0,               // CEveText draw handler not used
  0,               // EveNumber draw handler not used
  0                // CEveNumber draw handler not used
};
// ~Table of draw handler pointers

// Table of animations
void *ScreensEveAnimationTable[2] = {
  &EveSpinner1,
  0
};
// ~Table of animations


// Default configuration parameters
const TFT800Display VTFT_FT800_CONFIG_DISPLAY =
{
  48000000,        // Frequency          = main clock frequency
  0,               // OutRenderMode      = 0 normal, 1 write, 2 read
  0,               // RenderReadScanLine = scanline for read render mode
  0,               // RenderWriteTrigger = trigger for write render mode (read only)
  270,             // hCycle             = number if horizontal cycles for display
  20,              // hOffset            = horizontal offset from starting signal
  240,             // hSize              = width resolution
  0,               // hSync0             = hsync falls
  10,              // hSync1             = hsync rise
  326,             // vCycle             = number of vertical cycles for display
  2,               // vOffset            = vertical offset from start signal
  320,             // vSize              = height resolution
  0,               // vSync0             = vsync falls
  2,               // vSync1             = vsync rise
  0,               // Rotate             = rotate display
  0x01B6,          // OutBits            = output bits resolution
  0,               // OutDither          = output number of bits
  0x0004,          // OutSwizzle         = output swizzle
  0,               // OutCSpread         = output clock spread enable
  1,               // PClockPolarity     = clock polarity: 0 - rising edge, 1 - falling edge
  5,               // PClock             = clock prescaler of FT800: - 0 means disable and >0 means 48MHz/pclock
};

const TFT800GPIO VTFT_FT800_CONFIG_GPIO =
{
  0x83,            // GPIODIR = GPIO direction: 1 - output, 0 - input (8bit wide)
  0x83,            // GPIO    = GPIO data latch
};

const TFT800PWM VTFT_FT800_CONFIG_PWM =
{
  250,             // Freq = PWM frequency - 14 bits
  80,              // Duty = PWM duty cycle, 0 to 128 is the range
};

const TFT800Interrupt VTFT_FT800_CONFIG_INTERRUPT =
{
  0,               // Flags  = interrupt flags (read only)
  0,               // Enable = global interrupt enable: 1 - enabled, 0 - disabled
  255,             // Mask   = interrupt mask value (individual interrupt enable): 1 - masked/disabled, 0 - enabled
};

const TFT800Sound VTFT_FT800_CONFIG_SOUND =
{
  0,               // Volume
  {0,              // Effect
  0},              // Pitch
  0,               // Play
};

const TFT800Audio VTFT_FT800_CONFIG_AUDIO =
{
  0,               // Volume
  0,               // StartAddress
  0,               // Length
  0,               // ReadPtr
  8000,            // Frequency
  0,               // Format
  0,               // Loop = audio playback mode
  0,               // Play
};

// Global variables

TTouchStat TouchS = {0};


/////////////////////////
TScreen *CurrentScreen = 0;


TScreen Screen1;

TCircle     Circle1;
TEvent      Circle1_OnClick;
TCircle     Circle4;
TEvent      Circle4_OnClick;
TBox        Box1;
TEvent      Box1_OnClick;
TBox        Box2;
TCircle     Circle3;
TEvent      Circle3_OnClick;
TCircle     Circle2;
TEvent      Circle2_OnClick;
TEveSpinner EveSpinner1;

TCircle *const code Screen1_Circles[4] = {
  &Circle1,             
  &Circle4,             
  &Circle3,             
  &Circle2              
};

TBox *const code Screen1_Boxes[2] = {
  &Box1,                
  &Box2                 
};

TEveSpinner *const code Screen1_EveSpinners[1] = {
  &EveSpinner1          
};


TScreen Screen2;

TImage Image1;
TEvent Image1_OnClick;

TImage *const code Screen2_Images[1] = {
  &Image1               
};


static char IsInsideObject(TObjInfo *AObjInfo, unsigned int X, unsigned int Y) {
  TRect *ptrPressRect = 0;
  TRect *ptrPressCircle = 0;

  if ((AObjInfo->HitX == X) && (AObjInfo->HitY == Y))
    return 1;

  switch (AObjInfo->Type) {
    // Image
    case VTFT_OT_IMAGE: {
      ptrPressRect = (TRect *)&(((TImage *)AObjInfo->Obj)->Left);
      break;
    }
    // Circle
    case VTFT_OT_CIRCLE: {
      ptrPressCircle = (TRect *)&(((TCircle *)AObjInfo->Obj)->Left);
      break;
    }
    // Box
    case VTFT_OT_BOX: {
      ptrPressRect = (TRect *)&(((TBox *)AObjInfo->Obj)->Left);
      break;
    }
  }

  if (ptrPressRect) {
    if ((ptrPressRect->Left <= X) && (ptrPressRect->Left+ptrPressRect->Width-1 >= X) &&
        (ptrPressRect->Top  <= Y) && (ptrPressRect->Top+ptrPressRect->Height-1 >= Y))
      return 1;
    }
  else if (ptrPressCircle) {
    if ((ptrPressCircle->Left <= X) && (ptrPressCircle->Left+ptrPressCircle->Width*2-1 >= X) &&
        (ptrPressCircle->Top  <= Y) && (ptrPressCircle->Top+ptrPressCircle->Width*2-1 >= Y))
      return 1;
    }

  return 0;
}

void DrawImage(TImage *AImage) {
  TFT800BmpConfig bmpCfg;
  TFT800Rect currRect;
  TFT800Point rotationCenter;
  char format;
  unsigned int width, height;

  if (AImage->Visible) {
    if (AImage->Tag)
      FT800_Canvas_Tag(AImage->Tag);

    if (AImage->Picture_Type != _FT800_BITMAP_FORMAT_JPEG) {
      FT800_Canvas_BmpGetInfo(AImage->Picture_Name, &format, &width, &height);
      FT800_Canvas_BmpParamsToCfg(AImage->Source, width, height, format, &bmpCfg);
      FT800_Canvas_Bitmap(0, &bmpCfg, AImage->Blend_Color, AImage->Opacity);
      FT800_Canvas_PixelFormat12p4();
      FT800_Canvas_CurrRectSet(FT800_Canvas_IntTo12p4(AImage->Left), FT800_Canvas_IntTo12p4(AImage->Top), FT800_Canvas_IntTo12p4(width), FT800_Canvas_IntTo12p4(height));
      FT800_Canvas_Transform_Identity();
      rotationCenter.X = FT800_Canvas_IntTo12p4(AImage->RotationCenterLeft);
      rotationCenter.Y = FT800_Canvas_IntTo12p4(AImage->RotationCenterTop);
      FT800_Canvas_Transform_RotateAndScale(AImage->RotationAngle, &rotationCenter, AImage->ScaleX * 65536, AImage->ScaleY * 65536);
      FT800_Canvas_Transform_Set();
      FT800_Canvas_CurrRectGet(&currRect.Left, &currRect.Top, &currRect.Width, &currRect.Height);

      bmpCfg.Size.Width  = FT800_Canvas_12p4ToInt(currRect.Width);
      bmpCfg.Size.Height = FT800_Canvas_12p4ToInt(currRect.Height);
      FT800_Canvas_Bitmap(0, &bmpCfg, AImage->Blend_Color, AImage->Opacity);
      FT800_Screen_Bitmap(currRect.Left, currRect.Top);
      FT800_Canvas_Transform_Identity();
      FT800_Canvas_Transform_Set();
      FT800_Canvas_PixelFormatInt();
    }
  }
}

void DrawCircle(TCircle *ACircle) {
  if (ACircle->Visible) {
    if (ACircle == (TCircle *)TouchS.ActObjInfo.Obj)
      FT800_Canvas_BrushSingleColor(_FT800_BRUSH_STYLE_SOLID, ACircle->Press_Color, ACircle->Opacity);
    else 
      FT800_Canvas_BrushSingleColor(_FT800_BRUSH_STYLE_SOLID, ACircle->Color, ACircle->Opacity);

    FT800_Canvas_Pen(ACircle->Pen_Width, ACircle->Pen_Color, ACircle->Opacity);

    if (ACircle->Tag)
      FT800_Canvas_Tag(ACircle->Tag);

    FT800_Screen_Circle(ACircle->Left+ACircle->Radius, 
                      ACircle->Top+ACircle->Radius, 
                      ACircle->Radius);
  }
}

void DrawBox(TBox *ABox) {
  if (ABox->Visible) {
    if (ABox == (TBox *)TouchS.ActObjInfo.Obj) {
      if (ABox->Gradient != _FT800_BRUSH_GR_NONE) {
        FT800_Canvas_BrushGradient(_FT800_BRUSH_STYLE_SOLID, ABox->Gradient, ABox->Press_Color, ABox->Press_ColorTo, ABox->Opacity);
      }
      else {
        FT800_Canvas_BrushSingleColor(_FT800_BRUSH_STYLE_SOLID, ABox->Press_Color, ABox->Opacity);
      }
    }
    else {
      if (ABox->Gradient != _FT800_BRUSH_GR_NONE) {
        FT800_Canvas_BrushGradient(_FT800_BRUSH_STYLE_SOLID, ABox->Gradient, ABox->Color, ABox->ColorTo, ABox->Opacity);
      }
      else {
        FT800_Canvas_BrushSingleColor(_FT800_BRUSH_STYLE_SOLID, ABox->Color, ABox->Opacity);
      }
    }

    FT800_Canvas_Pen(ABox->Pen_Width, ABox->Pen_Color, ABox->Opacity);

    if (ABox->Tag)
      FT800_Canvas_Tag(ABox->Tag);

    FT800_Screen_Box(ABox->Left, ABox->Top, ABox->Left+ABox->Width-1, ABox->Top+ABox->Height-1);
  }
}

char DrawEveSpinner(TEveSpinner *AEveSpinner) {
  if(*CurrentScreen->EveAnimation == (unsigned short*)AEveSpinner) {
    if (AEveSpinner->Tag)
      FT800_Canvas_Tag(AEveSpinner->Tag);

    FT800_Canvas_BrushSingleColor(_FT800_BRUSH_STYLE_SOLID, AEveSpinner->Color, 255);

    FT800_Canvas_CPGraphics_Spinner(AEveSpinner->Style, AEveSpinner->Scale);
    FT800_Screen_SpinnerCP(AEveSpinner->Left, AEveSpinner->Top);
    return 1;
  }
  return 0;
}

void DisableEveAnimation(TScreen *AScreen) {
  *AScreen->EveAnimation = 0;
}

void EnableEveAnimation(TScreen *AScreen, void * AEveAnimation) {
  *AScreen->EveAnimation = AEveAnimation;
}

void SetIdenticalImageSources(const code char *APictureName, unsigned long ASource) {
  char i;
  TImage *pImage;
  void *const code *ptrO;

  // Image
  i    = CurrentScreen->ImagesCount;
  ptrO = CurrentScreen->Images;
  while (i--) {
    pImage = (TImage *)(*ptrO);
    if (pImage->Picture_Name == APictureName)
      pImage->Source = ASource;
    ptrO++;
  }
}

static void ClearDynObjSource() {
  char i;
  TImage *pImage;
  void *const code *ptrO;

  // Image
  i    = CurrentScreen->ImagesCount;
  ptrO = CurrentScreen->Images;
  while (i--) {
    pImage = (TImage *)(*ptrO);
    pImage->Source = 0xFFFFFFFF;
    ptrO++;
  }
}

void LoadCurrentScreenResToGRAM(char loadOptions) {
  char i;
  long currSource = -1;
  long tmpDynResStart;
  TImage *pImage;
  void *const code *ptrO;

  // clear dynamic resource addresses first, if necessary
  if (loadOptions & VTFT_LOAD_RES_DYNAMIC)
    ClearDynObjSource();

  // dynamic resources allocation
  if (loadOptions & VTFT_LOAD_RES_DYNAMIC) {

    tmpDynResStart = CurrentScreen->DynResStart;
    if (FT800_RES_BeginLoad(tmpDynResStart)) {
      return;
    }

    // Image
    i    = CurrentScreen->ImagesCount;
    ptrO = CurrentScreen->Images;
    while (i--) {
      pImage = (TImage *)(*ptrO);
      if (pImage->Source == 0xFFFFFFFF) {
        currSource = FT800_RES_LoadBitmap(pImage->Picture_Name);
        pImage->Source = currSource;
        SetIdenticalImageSources(pImage->Picture_Name, currSource);
      }
      ptrO++;
    }

    FT800_RES_EndLoad();
  }
}

void DrawObject(TPointer aObj, char aObjType) {
  TDrawHandler drawHandler;

  drawHandler = DrawHandlerTable[aObjType];
  if (drawHandler)
    drawHandler(aObj);
}

void DrawScreenO(TScreen *aScreen, char aOptions) {
  unsigned short cOrder, saveOrder;
  signed   int   actObjOrder;
  unsigned short pwmDuty;
  // counter variables
  char cntImage;
  char cntCircle;
  char cntBox;
  char cntEveSpinner;
  // pointer variables
  TImage      *const code *pImage;
  TCircle     *const code *pCircle;
  TBox        *const code *pBox;
  TEveSpinner *const code *pEveSpinner;

  // process screen switching effects
  if (aOptions & VTFT_DISPLAY_EFF_LIGHTS_FADE) {
    FT800_PWM_Get(0, &pwmDuty);
    FT800_PWM_FadeOut(pwmDuty, 0, pwmDuty/32? pwmDuty/32 : 1, 1);
  }
  else if (aOptions & VTFT_DISPLAY_EFF_LIGHTS_OFF) {
    FT800_PWM_Get(0, &pwmDuty);
    FT800_PWM_Duty(0);
  }

  if (CurrentScreen != aScreen) {
  // clear active object when drawing to new screen
    memset(&TouchS.ActObjInfo, 0, sizeof(TObjInfo));
  }

  CurrentScreen = aScreen;

  LoadCurrentScreenResToGRAM(aOptions);

  // init counter variables
  cntImage      = CurrentScreen->ImagesCount;
  cntCircle     = CurrentScreen->CirclesCount;
  cntBox        = CurrentScreen->BoxesCount;
  cntEveSpinner = CurrentScreen->EveSpinnersCount;
  // init pointer variables
  pImage      = CurrentScreen->Images;
  pCircle     = CurrentScreen->Circles;
  pBox        = CurrentScreen->Boxes;
  pEveSpinner = CurrentScreen->EveSpinners;

  FT800_Screen_BeginUpdateCP();
  FT800_Canvas_BrushSingleColor(_FT800_BRUSH_STYLE_SOLID, CurrentScreen->Color, 255);
  FT800_Canvas_Tag(0);
  FT800_Screen_Clear(_FT800_CLEAR_ALL);
  FT800_CP_CmdStop();

  actObjOrder = -1;
  if (TouchS.ActObjInfo.Obj)
    if (TouchS.ActObjInfo.Flags & VTFT_INT_BRING_TO_FRONT)
      actObjOrder = TouchS.ActObjInfo.Order;

  cOrder = 0;
  while (cOrder < CurrentScreen->ObjectsCount) {
    saveOrder = cOrder;
    if (pImage) {
      while ((*pImage)->Order == cOrder) {
        if (actObjOrder != cOrder) // draw pressed object at the end
          DrawImage(*pImage);
        cOrder++;
        pImage++;
        cntImage--;
        if (!cntImage) {
          pImage = 0;
          break;
        }
      }
      if (saveOrder != cOrder)
        continue;
    }

    if (pCircle) {
      while ((*pCircle)->Order == cOrder) {
        if (actObjOrder != cOrder) // draw pressed object at the end
          DrawCircle(*pCircle);
        cOrder++;
        pCircle++;
        cntCircle--;
        if (!cntCircle) {
          pCircle = 0;
          break;
        }
      }
      if (saveOrder != cOrder)
        continue;
    }

    if (pBox) {
      while ((*pBox)->Order == cOrder) {
        if (actObjOrder != cOrder) // draw pressed object at the end
          DrawBox(*pBox);
        cOrder++;
        pBox++;
        cntBox--;
        if (!cntBox) {
          pBox = 0;
          break;
        }
      }
      if (saveOrder != cOrder)
        continue;
    }

    cOrder++;
  }

  // draw pressed object now
  if (TouchS.ActObjInfo.Obj)
    DrawObject(TouchS.ActObjInfo.Obj, TouchS.ActObjInfo.Type);

  while(cntEveSpinner > 0) {
    if(DrawEveSpinner(*pEveSpinner) != 0) {
      break;
    }
    pEveSpinner++;
    cntEveSpinner--;
  }

  if(cntEveSpinner == 0) {
    FT800_Screen_EndUpdate();
    FT800_Screen_Show();
  }
  else {
    FT800_CP_EndUpdate();
    FT800_Display_WaitSwapComplete(2000);
  }

  // process screen switching effects
  if (aOptions & VTFT_DISPLAY_EFF_LIGHTS_FADE) {
    FT800_PWM_FadeIn(0, pwmDuty, 1, 3);
  }
  else if (aOptions & VTFT_DISPLAY_EFF_LIGHTS_OFF) {
    FT800_PWM_Duty(pwmDuty);
  }

}

void DrawScreen(TScreen *aScreen) {
  if (aScreen != CurrentScreen)
    DrawScreenO(aScreen, VTFT_LOAD_RES_ALL | VTFT_DISPLAY_EFF_LIGHTS_FADE);
  else
    DrawScreenO(aScreen, VTFT_LOAD_RES_NONE);
}

char GetActiveObjectByXY(int X, int Y, TObjInfo *AObjInfo) {
  char i;
  int  hiOrder;
  TImage      *pImage;
  TCircle     *pCircle;
  TBox        *pBox;
  void *const code *ptrO;

  // clear current object info
  memset(AObjInfo, 0, sizeof(TObjInfo));

  // Find object with highest order at specified position.
  // Objects lists are sorted by object order ascending.
  hiOrder = -1;

  // Image
  i    = CurrentScreen->ImagesCount;
  ptrO = CurrentScreen->Images+CurrentScreen->ImagesCount-1;
  while (i--) {
    pImage = (TImage *)(*ptrO);
    if (pImage->Order < hiOrder)
      break;
    if (pImage->Active) {
      if ((pImage->Left <= X) && (pImage->Left+pImage->Width-1 >= X) &&
          (pImage->Top  <= Y) && (pImage->Top+pImage->Height-1 >= Y)) {
        AObjInfo->Obj   = (void *)pImage;
        AObjInfo->Type  = VTFT_OT_IMAGE;
        AObjInfo->Order = pImage->Order;
        AObjInfo->Flags = VTFT_INT_BRING_TO_FRONT;

        hiOrder         = pImage->Order;

        break;
      }
    }
    ptrO--;
  }

  // Circle
  i    = CurrentScreen->CirclesCount;
  ptrO = CurrentScreen->Circles+CurrentScreen->CirclesCount-1;
  while (i--) {
    pCircle = (TCircle *)(*ptrO);
    if (pCircle->Order < hiOrder)
      break;
    if (pCircle->Active) {
      // for circle objects common object width is its radius property
      if ((pCircle->Left <= X) && (pCircle->Left+pCircle->Radius*2-1 >= X) &&
          (pCircle->Top  <= Y) && (pCircle->Top+pCircle->Radius*2-1 >= Y)) {
        AObjInfo->Obj   = (void *)pCircle;
        AObjInfo->Type  = VTFT_OT_CIRCLE;
        AObjInfo->Order = pCircle->Order;
        AObjInfo->Flags = VTFT_INT_BRING_TO_FRONT;
        if (pCircle->Press_Color != pCircle->Color)
          AObjInfo->Flags |= VTFT_INT_REPAINT_ON_DOWN | VTFT_INT_REPAINT_ON_UP;

        hiOrder         = pCircle->Order;

        break;
      }
    }
    ptrO--;
  }

  // Box
  i    = CurrentScreen->BoxesCount;
  ptrO = CurrentScreen->Boxes+CurrentScreen->BoxesCount-1;
  while (i--) {
    pBox = (TBox *)(*ptrO);
    if (pBox->Order < hiOrder)
      break;
    if (pBox->Active) {
      if ((pBox->Left <= X) && (pBox->Left+pBox->Width-1 >= X) &&
          (pBox->Top  <= Y) && (pBox->Top+pBox->Height-1 >= Y)) {
        AObjInfo->Obj   = (void *)pBox;
        AObjInfo->Type  = VTFT_OT_BOX;
        AObjInfo->Order = pBox->Order;
        AObjInfo->Flags = VTFT_INT_BRING_TO_FRONT;
        if ((pBox->Press_Color != pBox->Color) ||
            ((pBox->Gradient != _FT800_BRUSH_GR_NONE) &&
             (pBox->Press_ColorTo != pBox->ColorTo)))
          AObjInfo->Flags |= VTFT_INT_REPAINT_ON_DOWN | VTFT_INT_REPAINT_ON_UP;

        hiOrder         = pBox->Order;

        break;
      }
    }
    ptrO--;
  }

  if (AObjInfo->Obj) {
    AObjInfo->HitX = X;
    AObjInfo->HitY = Y;
    return 1;
  }
  else {
    return 0;
  }
}

char GetActiveObjectByTag(char ATag, TObjInfo *AObjInfo) {
  char i;
  TImage      *pImage;
  TCircle     *pCircle;
  TBox        *pBox;
  void *const code *ptrO;

  // clear current object info
  memset(AObjInfo, 0, sizeof(TObjInfo));

  // Find object with specified tag value.

  // Image
  i    = CurrentScreen->ImagesCount;
  ptrO = CurrentScreen->Images+CurrentScreen->ImagesCount-1;
  while (i--) {
    pImage = (TImage *)(*ptrO);
    if (pImage->Tag == ATag) {
      if (pImage->Active) {
        AObjInfo->Obj   = (void *)pImage;
        AObjInfo->Type  = VTFT_OT_IMAGE;
        AObjInfo->Order = pImage->Order;
        AObjInfo->Flags = VTFT_INT_BRING_TO_FRONT;
      }
      break;
    }
    ptrO--;
  }

  // Circle
  i    = CurrentScreen->CirclesCount;
  ptrO = CurrentScreen->Circles+CurrentScreen->CirclesCount-1;
  while (i--) {
    pCircle = (TCircle *)(*ptrO);
    if (pCircle->Tag == ATag) {
      if (pCircle->Active) {
        AObjInfo->Obj   = (void *)pCircle;
        AObjInfo->Type  = VTFT_OT_CIRCLE;
        AObjInfo->Order = pCircle->Order;
        AObjInfo->Flags = VTFT_INT_BRING_TO_FRONT;
        if (pCircle->Press_Color != pCircle->Color)
          AObjInfo->Flags |= VTFT_INT_REPAINT_ON_DOWN | VTFT_INT_REPAINT_ON_UP;

      }
      break;
    }
    ptrO--;
  }

  // Box
  i    = CurrentScreen->BoxesCount;
  ptrO = CurrentScreen->Boxes+CurrentScreen->BoxesCount-1;
  while (i--) {
    pBox = (TBox *)(*ptrO);
    if (pBox->Tag == ATag) {
      if (pBox->Active) {
        AObjInfo->Obj   = (void *)pBox;
        AObjInfo->Type  = VTFT_OT_BOX;
        AObjInfo->Order = pBox->Order;
        AObjInfo->Flags = VTFT_INT_BRING_TO_FRONT;
        if ((pBox->Press_Color != pBox->Color) ||
            ((pBox->Gradient != _FT800_BRUSH_GR_NONE) &&
             (pBox->Press_ColorTo != pBox->ColorTo)))
          AObjInfo->Flags |= VTFT_INT_REPAINT_ON_DOWN | VTFT_INT_REPAINT_ON_UP;

      }
      break;
    }
    ptrO--;
  }

  if (AObjInfo->Obj) {
    AObjInfo->HitTag = ATag;
    return 1;
  }
  else {
    return 0;
  }
}

static void ProcessEvent(TEvent *pEvent) {
  if (pEvent) {
    if (pEvent->Sound.SndAct == VTFT_SNDACT_PLAY) 
      FT800_Sound_SetAndPlay(pEvent->Sound.Effect, pEvent->Sound.Pitch, pEvent->Sound.Volume);
    else if (pEvent->Sound.SndAct == VTFT_SNDACT_STOP) 
      FT800_Sound_Stop();
    if (pEvent->Action) 
      pEvent->Action();
  }
}

static void ProcessCEvent(TCEvent *pEventC) {
  if (pEventC) {
    if (pEventC->Sound.SndAct == VTFT_SNDACT_PLAY) 
      FT800_Sound_SetAndPlay(pEventC->Sound.Effect, pEventC->Sound.Pitch, pEventC->Sound.Volume);
    else if (pEventC->Sound.SndAct == VTFT_SNDACT_STOP) 
      FT800_Sound_Stop();
    if (pEventC->Action) 
      pEventC->Action();
  }
}

static void OnEvent(TObjInfo *AObjInfo, char AEventType){
  TEvent **ppEvent;
  TEvent  *pEvent = 0;

  switch (AObjInfo->Type) {
    // Image
    case VTFT_OT_IMAGE: {
      ppEvent = &(((TImage *)AObjInfo->Obj)->OnUp);
      pEvent  = ppEvent[AEventType];
      break;
    }
    // Circle
    case VTFT_OT_CIRCLE: {
      ppEvent = &(((TCircle *)AObjInfo->Obj)->OnUp);
      pEvent  = ppEvent[AEventType];
      break;
    }
    // Box
    case VTFT_OT_BOX: {
      ppEvent = &(((TBox *)AObjInfo->Obj)->OnUp);
      pEvent  = ppEvent[AEventType];
      break;
    }
  } // end switch

  if (pEvent) {
    ProcessEvent(pEvent);
  }
}

static void Process_TP_Press() {
  // Screen Event
  if (CurrentScreen->Active)
    if ((CurrentScreen->SniffObjectEvents) || (!TouchS.ActObjInfo.Obj))
      ProcessEvent(CurrentScreen->OnPress);

  // Object Event
  if (!TouchS.ActObjInfo.Obj)
    return;

  OnEvent(&TouchS.ActObjInfo, VTFT_EVT_PRESS);
}

static void Process_TP_Up() {
  char     isClick;
  TObjInfo actObj;

  // Screen Event
  if (CurrentScreen->Active)
    if ((CurrentScreen->SniffObjectEvents) || (!TouchS.ActObjInfo.Obj))
      ProcessEvent(CurrentScreen->OnUp);

  actObj = TouchS.ActObjInfo;
  // Cler active object info
  memset(&TouchS.ActObjInfo, 0, sizeof(TObjInfo));

  // Object Event
  if (!actObj.Obj)
    return;

  isClick = IsInsideObject(&actObj, TouchS.X, TouchS.Y);

  if (actObj.Flags & VTFT_INT_REPAINT_ON_UP)
    DrawScreen(CurrentScreen);

  OnEvent(&actObj, VTFT_EVT_UP);
  if (isClick)
    OnEvent(&actObj, VTFT_EVT_CLICK);
}

static void Process_TP_Down() {
  // Search for active object
  if (TouchS.Tag) {        // objects must not have zero for tag value
    if (TouchS.Tag != 255) // can not search objects by default tag value
      GetActiveObjectByTag(TouchS.Tag, &TouchS.ActObjInfo);
    if (!TouchS.ActObjInfo.Obj) // object not found by tag, search by coordinates
      GetActiveObjectByXY(TouchS.X, TouchS.Y, &TouchS.ActObjInfo);
  }

  // Screen Event
  if (CurrentScreen->Active)
    if ((CurrentScreen->SniffObjectEvents) || (!TouchS.ActObjInfo.Obj))
      ProcessEvent(CurrentScreen->OnDown);

  // Object Event
  if (!TouchS.ActObjInfo.Obj)
    return;

  if (TouchS.ActObjInfo.Flags & VTFT_INT_REPAINT_ON_DOWN)
    DrawScreen(CurrentScreen);

  OnEvent(&TouchS.ActObjInfo, VTFT_EVT_DOWN);
}

static void Process_TP_TagChange() {
  // Screen Event
  if (CurrentScreen->Active)
    ProcessEvent(CurrentScreen->OnTagChange);
}

void ProcessVTFTStack() {
  char         Tag, oldTag;
  unsigned int X, Y;

  oldTag = TouchS.Tag;

  if (FT800_Touch_GetTagXY(&X, &Y) == 1) {
    FT800_Touch_GetTag(&Tag);

    TouchS.Tag = Tag;
    TouchS.X = X;
    TouchS.Y = Y;

    if (!TouchS.Pressed) {
      TouchS.Pressed = 1;
      Process_TP_Down();
    }

    Process_TP_Press();
  }
  else if (TouchS.Pressed) {
    Process_TP_Up();

    TouchS.Tag = 0;
    TouchS.X   = X;
    TouchS.Y   = Y;

    TouchS.Pressed = 0;
  }

  if (oldTag != TouchS.Tag)
    Process_TP_TagChange();
}

static void InitObjects() {
  // Screen1: Init block start
  Screen1.Color             = 0x0193;
  Screen1.Width             = 240;
  Screen1.Height            = 320;
  Screen1.ObjectsCount      = 7;
  Screen1.ImagesCount       = 0;
  Screen1.Images            = 0;
  Screen1.CirclesCount      = 4;
  Screen1.Circles           = Screen1_Circles;
  Screen1.BoxesCount        = 2;
  Screen1.Boxes             = Screen1_Boxes;
  Screen1.EveSpinnersCount  = 1;
  Screen1.EveSpinners       = Screen1_EveSpinners;
  Screen1.DynResStart       = 0;
  Screen1.Active            = 1;
  Screen1.EveAnimation      = &ScreensEveAnimationTable[0];
  Screen1.SniffObjectEvents = 0;
  Screen1.OnUp              = 0;
  Screen1.OnDown            = 0;
  Screen1.OnTagChange       = 0;
  Screen1.OnPress           = 0;

  // Screen2: Init block start
  Screen2.Color             = 0x0000;
  Screen2.Width             = 240;
  Screen2.Height            = 320;
  Screen2.ObjectsCount      = 1;
  Screen2.ImagesCount       = 1;
  Screen2.Images            = Screen2_Images;
  Screen2.CirclesCount      = 0;
  Screen2.Circles           = 0;
  Screen2.BoxesCount        = 0;
  Screen2.Boxes             = 0;
  Screen2.EveSpinnersCount  = 0;
  Screen2.EveSpinners       = 0;
  Screen2.DynResStart       = 0;
  Screen2.Active            = 1;
  Screen2.EveAnimation      = &ScreensEveAnimationTable[1];
  Screen2.SniffObjectEvents = 0;
  Screen2.OnUp              = 0;
  Screen2.OnDown            = 0;
  Screen2.OnTagChange       = 0;
  Screen2.OnPress           = 0;

  Circle1.OwnerScreen   = &Screen1;
  Circle1.Order         = 0;
  Circle1.Visible       = 1;
  Circle1.Opacity       = 255;
  Circle1.Tag           = 255;
  Circle1.Left          = 44;
  Circle1.Top           = 50;
  Circle1.Radius        = 80;
  Circle1.Pen_Width     = 5;
  Circle1.Pen_Color     = 0x0000;
  Circle1.Color         = 0xFFE0;
  Circle1.Press_Color   = 0xFFE0;
  Circle1.Active        = 0;
  Circle1.OnUp          = 0;
  Circle1.OnDown        = 0;
  Circle1.OnClick       = &Circle1_OnClick;
  Circle1.OnPress       = 0;

  Circle1_OnClick.Action       = R_Img;
  Circle1_OnClick.Sound.SndAct = VTFT_SNDACT_NONE;
  Circle1_OnClick.Sound.Effect = _FT800_SOUND_XYLOPHONE;
  Circle1_OnClick.Sound.Pitch  = _FT800_SOUND_PITCH_A5;
  Circle1_OnClick.Sound.Volume = 255;

  Circle4.OwnerScreen   = &Screen1;
  Circle4.Order         = 1;
  Circle4.Visible       = 1;
  Circle4.Opacity       = 255;
  Circle4.Tag           = 255;
  Circle4.Left          = 66;
  Circle4.Top           = 72;
  Circle4.Radius        = 58;
  Circle4.Pen_Width     = 5;
  Circle4.Pen_Color     = 0x0000;
  Circle4.Color         = 0xFFE0;
  Circle4.Press_Color   = 0xFFE0;
  Circle4.Active        = 1;
  Circle4.OnUp          = 0;
  Circle4.OnDown        = 0;
  Circle4.OnClick       = &Circle4_OnClick;
  Circle4.OnPress       = 0;

  Circle4_OnClick.Action       = R_Img;
  Circle4_OnClick.Sound.SndAct = VTFT_SNDACT_NONE;
  Circle4_OnClick.Sound.Effect = _FT800_SOUND_XYLOPHONE;
  Circle4_OnClick.Sound.Pitch  = _FT800_SOUND_PITCH_A5;
  Circle4_OnClick.Sound.Volume = 255;

  Box1.OwnerScreen   = &Screen1;
  Box1.Order         = 2;
  Box1.Visible       = 1;
  Box1.Opacity       = 255;
  Box1.Tag           = 255;
  Box1.Left          = 61;
  Box1.Top           = 89;
  Box1.Width         = 127;
  Box1.Height        = 47;
  Box1.Pen_Width     = 0;
  Box1.Pen_Color     = 0xFFE0;
  Box1.Color         = 0xFFE0;
  Box1.Press_Color   = 0xE71C;
  Box1.ColorTo       = 0xFFFF;
  Box1.Press_ColorTo = 0xC618;
  Box1.Gradient      = _FT800_BRUSH_GR_NONE;
  Box1.Active        = 0;
  Box1.OnUp          = 0;
  Box1.OnDown        = 0;
  Box1.OnClick       = &Box1_OnClick;
  Box1.OnPress       = 0;

  Box1_OnClick.Action       = R_Img;
  Box1_OnClick.Sound.SndAct = VTFT_SNDACT_NONE;
  Box1_OnClick.Sound.Effect = _FT800_SOUND_XYLOPHONE;
  Box1_OnClick.Sound.Pitch  = _FT800_SOUND_PITCH_A5;
  Box1_OnClick.Sound.Volume = 255;

  Box2.OwnerScreen   = &Screen1;
  Box2.Order         = 3;
  Box2.Visible       = 1;
  Box2.Opacity       = 255;
  Box2.Tag           = 255;
  Box2.Left          = 77;
  Box2.Top           = 68;
  Box2.Width         = 94;
  Box2.Height        = 27;
  Box2.Pen_Width     = 0;
  Box2.Pen_Color     = 0xFFE0;
  Box2.Color         = 0xFFE0;
  Box2.Press_Color   = 0xE71C;
  Box2.ColorTo       = 0xFFFF;
  Box2.Press_ColorTo = 0xC618;
  Box2.Gradient      = _FT800_BRUSH_GR_NONE;
  Box2.Active        = 0;
  Box2.OnUp          = 0;
  Box2.OnDown        = 0;
  Box2.OnClick       = 0;
  Box2.OnPress       = 0;

  Circle3.OwnerScreen   = &Screen1;
  Circle3.Order         = 4;
  Circle3.Visible       = 1;
  Circle3.Opacity       = 255;
  Circle3.Tag           = 255;
  Circle3.Left          = 140;
  Circle3.Top           = 91;
  Circle3.Radius        = 14;
  Circle3.Pen_Width     = 1;
  Circle3.Pen_Color     = 0x0000;
  Circle3.Color         = 0x0000;
  Circle3.Press_Color   = 0x0000;
  Circle3.Active        = 1;
  Circle3.OnUp          = 0;
  Circle3.OnDown        = 0;
  Circle3.OnClick       = &Circle3_OnClick;
  Circle3.OnPress       = 0;

  Circle3_OnClick.Action       = R_Img;
  Circle3_OnClick.Sound.SndAct = VTFT_SNDACT_NONE;
  Circle3_OnClick.Sound.Effect = _FT800_SOUND_XYLOPHONE;
  Circle3_OnClick.Sound.Pitch  = _FT800_SOUND_PITCH_A5;
  Circle3_OnClick.Sound.Volume = 255;

  Circle2.OwnerScreen   = &Screen1;
  Circle2.Order         = 5;
  Circle2.Visible       = 1;
  Circle2.Opacity       = 255;
  Circle2.Tag           = 255;
  Circle2.Left          = 81;
  Circle2.Top           = 91;
  Circle2.Radius        = 14;
  Circle2.Pen_Width     = 1;
  Circle2.Pen_Color     = 0x0000;
  Circle2.Color         = 0x0000;
  Circle2.Press_Color   = 0x0000;
  Circle2.Active        = 1;
  Circle2.OnUp          = 0;
  Circle2.OnDown        = 0;
  Circle2.OnClick       = &Circle2_OnClick;
  Circle2.OnPress       = 0;

  Circle2_OnClick.Action       = R_Img;
  Circle2_OnClick.Sound.SndAct = VTFT_SNDACT_NONE;
  Circle2_OnClick.Sound.Effect = _FT800_SOUND_XYLOPHONE;
  Circle2_OnClick.Sound.Pitch  = _FT800_SOUND_PITCH_A5;
  Circle2_OnClick.Sound.Volume = 255;

  EveSpinner1.OwnerScreen   = &Screen1;
  EveSpinner1.Tag           = 255;
  EveSpinner1.Left          = 124;
  EveSpinner1.Top           = 245;
  EveSpinner1.Color         = 0x03DA;
  EveSpinner1.Style         = 3;
  EveSpinner1.Scale         = 0;

  Image1.OwnerScreen   = &Screen2;
  Image1.Order         = 0;
  Image1.Visible       = 1;
  Image1.Opacity       = 255;
  Image1.Tag           = 255;
  Image1.Left          = 0;
  Image1.Top           = 0;
  Image1.Width         = 240;
  Image1.Height        = 320;
  Image1.Picture_Name  = Riverdi_bkg2_bmp;
  Image1.Picture_Type  = 3;
  Image1.Picture_Ratio = 1;
  Image1.Blend_Color   = 0xFFFF;
  Image1.Source        = -1L;
  Image1.RotationAngle = 0;
  Image1.RotationCenterLeft = 120;
  Image1.RotationCenterTop = 160;
  Image1.ScaleX        = 1.0000;
  Image1.ScaleY        = 1.0000;
  Image1.Active        = 1;
  Image1.OnUp          = 0;
  Image1.OnDown        = 0;
  Image1.OnClick       = &Image1_OnClick;
  Image1.OnPress       = 0;

  Image1_OnClick.Action       = R_Home;
  Image1_OnClick.Sound.SndAct = VTFT_SNDACT_NONE;
  Image1_OnClick.Sound.Effect = _FT800_SOUND_XYLOPHONE;
  Image1_OnClick.Sound.Pitch  = _FT800_SOUND_PITCH_A5;
  Image1_OnClick.Sound.Volume = 255;
}

void Init_MCU()
{
  uint8_ft8 chipID;
  AD1PCFG = 0xFFFF;
  LATB = 0;
  LATG = 0;

  FT800_CS_Direction = 0;
  FT800_RST_Direction = 0;
  FT800_CS = 1;
  FT800_RST = 1;

  //IDLE 2 ACTIVEEEEEEEEEEEEEEEEEEEEEEEEEAEAEAEEAAAAAAAAAAEEEEEEEEEEEEAEAEAEAAAAAAAAAAEEEEEEEEEEEEEEEEEEEEEEEEEAEAEAEAAE!!!!!!!!!!!
  SPI2_Init_Advanced(_SPI_MASTER, _SPI_8_BIT, 8, _SPI_SS_DISABLE, _SPI_DATA_SAMPLE_MIDDLE, _SPI_CLK_IDLE_HIGH, _SPI_IDLE_2_ACTIVE);

  Delay_ms(1000);
  Ft_Gpu_Hal_Powercycle(true);
  FT800_Core_PowerMode(_FT800_PWR_MODE_ACTIVE);
  Delay_ms(20);
  FT800_Core_ClockSource(_FT800_CLK_SOURCE_INTERNAL);
  Delay_ms(10);
  FT800_Core_ClockPLL(_FT800_CLK_PLL_48MHz);
  Delay_ms(10);
  FT800_Core_Reset();
  //Chip ID = 7Ch
  while(chipID != 0x7C)
  {
    FT800_IO_ReadBuffAddr(_FT800_REG_ID, &chipID, 1);
  }
  Delay_ms(100);
  
  //Set the FT GPIO to ILI interface (ILI_CS and ILI_CS direction)
  FT800_GPIO_Set(0x83, 0x83);

  //USING ILI 9 bit in bitbang mode, MSSP will get disconnected...
  ILI_init();
  //Return pin control to the MSSP
  ON__SPI2CON_bit = 1;
}

void InitVTFTStack() {
  Init_MCU();

  SPI_Set_Active(SPI2_Read, SPI2_Write);

  // Init all dynamic objects
  InitObjects();

  // Init FT800 controller core and library stack
  //FT800_Init();

  FT800_Core_ClockSource(_FT800_CLK_SOURCE_INTERNAL);
  FT800_Core_ClockPLL(_FT800_CLK_PLL_48MHz);

  // Internal modules setup
  FT800_Display_SetConfig(&VTFT_FT800_CONFIG_DISPLAY);

  FT800_Audio_SetConfig(&VTFT_FT800_CONFIG_AUDIO);

  FT800_Sound_SetConfig(&VTFT_FT800_CONFIG_SOUND);

  FT800_Interrupt_SetConfig(&VTFT_FT800_CONFIG_INTERRUPT);

  FT800_PWM_SetConfig(&VTFT_FT800_CONFIG_PWM);

  FT800_GPIO_SetConfig(&VTFT_FT800_CONFIG_GPIO);

  FT800_Touch_Mode(_FT800_TOUCH_SMPL_MODE_CONTINUOUS, _FT800_TOUCH_OP_MODE_COMPATIBILITY);

  FT800_Touch_Calibrate(_FT800_FONT_HELVETICA_SIZE_10, "Touch blinking point on the screen!");

  // Draw start screen
  DrawScreen(&Screen1);
}